# Prelims ----------------------------------------------------------------


## Years concordance
projection_years <- c(2018:2050)
projection_years_id <- c(0: (length(projection_years)-1))

year_concord <- data.frame(year=projection_years_id,
                           year_actual=projection_years)


billion <- 1000000000

today_date <- Sys.Date()

custom_plot_margin <- theme(plot.margin = unit(1.75 * c(0.3, 0.3, 0.1, 0.3), "lines"))



# Read in some other data to enable analysis ---------------------------------------------------------------------

# discount rate for wealth by AWE
awe <- qread("./Input data/AWE_projections.qs") %>%
  select(year, awe_growth_factor_2018) %>%
  filter(year<=2050)


## HILDA grouped master data
hilda_grouped <- qread("./Input data/Intermediate input data/hilda_grouped_master.qs")





# Functions for tidying and arranging results --------------------------------------

## tidy and create some new vars
tidy_results_fn <- function(x) {
  x %>% 
    rbindlist(use.names=TRUE, fill=T) %>% 
    mutate(total_wealth = housing_assets + super_assets + other_assets - housing_debt) %>% 
    ## get actual year projection refers to
    left_join(year_concord) %>% 
    select(-year, year=year_actual) %>% 
    ## create a "generation" variable - groupings of age0
    mutate(gen0 = case_when(
      age_grp_0 >= "[75,80)" ~ "Pre-boomers", ## approximation -- until 1945 meaning youngest are 73 in 2018
      age_grp_0 >= "[55,60)" & age_grp_0 <= "[70,75)" ~ "Boomers", ## approximation -- 1946-64, meaning aged 54-72 in 2018
      age_grp_0 >= "[40,45)" & age_grp_0 <= "[50,55)" ~ "Gen X", ## 1965-80, aged 38-53 in 2018
      age_grp_0 >= "[20,25)" & age_grp_0 <= "[35,40)" ~ "Millennials", ## 1981-96, aged 22-37 in 2018
      age_grp_0 <= "[15,20)" ~ "Post-millennials" ## gen Z and gen alpha
    ) %>% 
      factor(levels=c("Pre-boomers", "Boomers", "Gen X", "Millennials", "Post-millennials") %>% rev, ordered=T)
    ) %>% 
    ## add in AWE growth factor for discounting to 2018 values
    left_join(awe) %>% 
    ## real wealth variables
    mutate(across(all_of(c("total_wealth", "housing_assets", "super_assets", "other_assets", "housing_debt", "total_wealth_at_death",
                           "av_beqrec", "av_giftrec", "av_giftgiven")),
                  ~.x/awe_growth_factor_2018,
                  .names = "{.col}_real")
    ) 
}


## new grouping vars - wealth group by age grp 0
inc_wealth_group_fn <- function(x) {
  x %>% 
    group_by(year, age_grp_0) %>% 
    group_split %>% 
    lapply(. , function(y) {
      y %>% 
        mutate(
          age_wlth_grp3 = ifelse(age_grp>="[30,35)" & age_grp<="[85,90)" & year>=2020,
                                 cut(total_wealth, 
                                     ## cut points determined by weighted quantile (only works if cut points are unique)
                                     breaks = wtd.quantile(.$total_wealth, 
                                                           weights = .$n, 
                                                           probs = seq(0, 1, 1/3)), 
                                     include.lowest=T,
                                     labels = c(1:3),
                                     ordered_result=T),
                                 NA)
        ) %>% 
        ## fix issue with sometimes missing age_wlth_grp3 at min or max values & rounding issues
        mutate(age_wlth_grp3 = case_when(is.na(age_wlth_grp3) & round(total_wealth, 2) == round( min(total_wealth), 2) ~ 1,
                                         is.na(age_wlth_grp3) & round(total_wealth, 2) == round( max(total_wealth), 2) ~ 3,
                                         TRUE ~ as.numeric(age_wlth_grp3) )) 
    }) %>% 
    rbindlist
}


## for use below in creating dataframes by group
summarise_vars_fn <- function(x) {
  x %>% 
    summarise(buckets = n(),
              n_tot = sum(n),
              deaths = sum(end_year_deaths),
              
              housing_assets_tot = sum(housing_assets*n),
              super_assets_tot = sum(super_assets*n),
              other_assets_tot = sum(other_assets*n),
              housing_debt_tot = -sum(housing_debt*n),
              total_wealth_tot = sum(total_wealth*n),
              
              housing_assets_tot_real = sum(housing_assets_real*n),
              super_assets_tot_real = sum(super_assets_real*n),
              other_assets_tot_real = sum(other_assets_real*n),
              housing_debt_tot_real = -sum(housing_debt_real*n),
              total_wealth_tot_real = sum(total_wealth_real*n),
              
              housing_assets_av = wtd.mean(housing_assets, weights=n),
              super_assets_av = wtd.mean(super_assets, weights=n),
              other_assets_av = wtd.mean(other_assets, weights=n),
              housing_debt_av = -wtd.mean(housing_debt, weights=n),
              total_wealth_av = wtd.mean(total_wealth, weights=n),
              
              housing_assets_av_real = housing_assets_av/mean(awe_growth_factor_2018),
              super_assets_av_real   = super_assets_av/mean(awe_growth_factor_2018),
              other_assets_av_real   = other_assets_av/mean(awe_growth_factor_2018),
              housing_debt_av_real   = housing_debt_av/mean(awe_growth_factor_2018),
              total_wealth_av_real   = total_wealth_av/mean(awe_growth_factor_2018),
              
              total_wealth_at_death_tot_real = sum( end_year_deaths*total_wealth_at_death_real ), 
              total_wealth_at_death_av_real = total_wealth_at_death_tot_real/deaths,
              
              bequest_given_intergen_tot_real = sum( end_year_deaths*total_wealth_at_death_real*(1-partnered_pc)*0.98 ) , ## 2% of intergen goes to charities
              
              bequest_received_tot = sum(av_beqrec*prev_parent_deaths_add, na.rm=T), ## Note prev_parent_deaths_add is number of parent deaths of those who are still living and just received a bequest
              bequest_received_tot_real = sum(av_beqrec_real*prev_parent_deaths_add, na.rm=T),
              bequest_received_av = wtd.mean(av_beqrec, weights=prev_parent_deaths_add, na.rm=T),
              bequest_received_av_real = bequest_received_av/mean(awe_growth_factor_2018),
              
              gift_given_tot = sum(av_giftgiven*n, na.rm=T) ,
              gift_given_tot_real = sum(av_giftgiven_real*n, na.rm=T) ,

              gift_received_tot = sum(av_giftrec*n, na.rm=T) , 
              gift_received_tot_real = sum(av_giftrec_real*n, na.rm=T) , 
              gift_received_av = wtd.mean(av_giftrec, weights=n) ,
              gift_received_av_real = gift_received_av/mean(awe_growth_factor_2018),
              
              ## bequest received as a share of previous wealth for new beqrecs - nominal/nominal
              beqrec_wealth_share = bequest_received_tot/sum(prev_wealth_new_beqrec*prev_parent_deaths_add, na.rm=T),
              
              ## bequest received as a share of MODEL lifetime inc up to that point -- you can really only compare this within same starting age group
              beqrec_model_inc_share = bequest_received_tot_real/sum(model_lifetime_inc_2018 * prev_parent_deaths_add, na.rm=T),
              
              ## average model lifetime income in 2018 dollars
              av_model_lifetime_inc_2018 = wtd.mean(model_lifetime_inc_2018, weights=n)
              
    )
}


## Converts age variable formatting from [a,b) to a-b for charts
age_grp_labeller <- function(x) {
  ## x is a string of age groups in [.,.) format.
  ## this function edits it so it is in .-. format
  first_num <- x %>% 
    str_extract("[[:digit:]]+")
  
  second_num <- x %>% 
    str_extract(",[[:digit:]]+") %>% 
    str_sub(2, -1) %>% 
    as.numeric 
  second_num <- second_num-1
  
  new_label <- paste0(first_num, "\U2013", second_num) %>% 
    str_replace(paste0("\U2013","104"), "+")
}

